#define _GNU_SOURCE

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <sys/wait.h>

void tsh_prompt();
int main_pgrp;

//=============== LIST ==============
struct p_list {
    struct p_node *head;
    struct p_node *tail;
};

struct p_node {
    pid_t pid;
    int jid;
    struct p_node *next;
};

struct p_node dummy;

struct p_list *
p_init() {
    struct p_list *list = malloc(sizeof(struct p_list));
    list->head = list->tail = NULL;
    return list;
}

int // jid is returned
p_append(struct p_list *list, pid_t pid) {
    struct p_node *e = malloc(sizeof(struct p_node));
    e->pid = pid;
    e->next = NULL;

    if (list->head == NULL) {
        list->head = e;
        list->tail = e;
        e->jid = 1;
        return 1;
    }

    struct p_node *p = list->tail;
    p->next = e;
    list->tail = e;
    e->jid = p->jid+1;

    return e->jid;
}

pid_t // pid is returned
j_remove(struct p_list *list, int jid) {
    struct p_node *e = list->head;
    if (e == NULL) return 0;

    struct p_node *p = NULL;
    while (e != NULL && e->jid != jid) {
        e = e->next;
        p = e;
    }

    if (e == NULL) return 0;
    if (p == NULL) { // rm from head
        list->head = e->next;
        if (e->next == NULL) { // rm from tail
            list->tail = NULL;
        }
    } else { // rm not from head
        p->next = e->next;
        if (e->next == NULL) { // rm from tail
            list->tail = NULL;
        }
    }

    dummy.next = e->next;
    pid_t pid = e->pid;

    free(e);
    return pid;
}

struct p_node * // a dummy node is return if success
p_remove(struct p_list *list, pid_t pid) {
    struct p_node *e = list->head;
    if (e == NULL) return NULL;

    struct p_node *p = NULL;
    while (e != NULL && e->pid != pid) {
        e = e->next;
        p = e;
    }

    if (e == NULL) return NULL;
    if (p == NULL) { // rm from head
        list->head = e->next;
        if (e->next == NULL) { // rm from tail
            list->tail = NULL;
        }
    } else { // rm not from head
        p->next = e->next;
        if (e->next == NULL) { // rm from tail
            list->tail = NULL;
        }
    }

    dummy.next = e->next;

    free(e);
    return &dummy;
}
//==================================

char cwd[1024];
volatile pid_t fgpid = 0;
struct p_list *bg_list;


void bg_update(pid_t pid, int status) {
    for (struct p_node *e = bg_list->head; e != NULL; e = e->next) {
        if (pid == 0) {
            printf ("[%d] %d\n", e->jid, e->pid);
        }
        else if (e->pid == pid) {
            if (WTERMSIG(status) == 9 || WIFEXITED(status) || WIFSIGNALED(status) || WIFCONTINUED(status)) {
                // remove e
                tcsetpgrp(STDIN_FILENO, main_pgrp);
                p_remove(bg_list, pid);
            }
            break;
        }
    }
}

//========== SIG HANDLER ===========
static void
sig_int(int signo) {
    if (fgpid > 0)
        kill(fgpid, signo);
    else {
        puts(""); tsh_prompt(); fflush(0);
    }
    fgpid = 0;
}

static void
sig_chld(int signo) {
    pid_t pid;
    int status;

    while ((pid = waitpid(-1, &status, WNOHANG|WUNTRACED)) > 0) {
        bg_update(pid, status);
    }
}

static void
sig_tstp(int signo) {
    puts(""); tsh_prompt(); fflush(0);
}

//==================================

void
command(char *line) {
    int tok_size = 16;
    char **token = malloc(tok_size * sizeof(char *));
    int i = 0;

    char *tok = strtok(line, " \t\r\n");
    while (tok != NULL) {
        token[i++] = tok;
        tok = strtok(NULL, " \t\r\n");

        if (i >= tok_size) {
            tok_size *= 2;
            token = realloc(token, tok_size * sizeof(char *));
        }
    }

    token[i] = NULL;

    if (i == 0) {
        return;
    }

    int isBG = 0;
    char *last_token = token[i-1];
    const int last_char_i = strlen(last_token)-1;
    if (last_token[last_char_i] == '&') {
        isBG = 1;

        if (last_char_i == 0) {
            token[i-1] = NULL;
        } else {
            last_token[last_char_i] = '\0';
        }
    }

    if ( strcmp(token[0], "exit") == 0 ) {
        exit(0);
    } else
    if ( strcmp(token[0], "pwd") == 0 ) {
        puts (getcwd(cwd, 1024));
    } else
    if ( strcmp(token[0], "jobs") == 0 ) {
        bg_update (0, 0);
    } else
    if ( strcmp(token[0], "fg") == 0 ) {
        if (i-1 >= 1) {
            fgpid = j_remove (bg_list, strtol(token[1], NULL, 10));
            if (fgpid == 0) {
                printf ("-tsh: fg: %s: no such job\n", token[1]);
            } else {
                tcsetpgrp(STDIN_FILENO, fgpid);
                waitpid(fgpid, NULL, 0);
                tcsetpgrp(STDIN_FILENO, main_pgrp);
            }
        } else {
            printf ("-tsh: fg: : no such job\n");
        }
    } else
    if ( strcmp(token[0], "cd") == 0 ) {
        if (i == 1) {
            if (chdir(getenv("HOME")) == -1) {
                printf ("-tsh: cd: $HOME is not set\n");
            }
        } else if (chdir (token[1]) == -1) {
            printf ("-tsh: %s: no such file or directory\n", token[1]);
        }
    } else { // launch a program
        if ((fgpid = fork()) == -1) {
            printf ("-tsh: child process creation failed\n");
        } else if (fgpid == 0) {
            // I am the child process!
            /*signal(SIGINT, sig_int);
            signal(SIGCHLD, sig_chld);
            signal(SIGTSTP, sig_tstp);*/
            pid_t pid = getpid();
            setpgid(pid, pid);

            if (execvp(token[0], token) == -1) {
                if (strchr(token[0], '/') != NULL)
                    printf ("-tsh: %s: No such file or directory\n", token[0]);
                else
                    printf ("%s: command not found\n", token[0]);
                free(token);
                exit(EXIT_FAILURE);
            }
        }

        free(token);
        // I am the parent process (tsh)!
        if (!isBG) {
            tcsetpgrp(STDIN_FILENO, fgpid);
            waitpid(fgpid, NULL, 0);
            tcsetpgrp(STDIN_FILENO, main_pgrp);
        } else {
            int jid = p_append(bg_list, fgpid);
            printf ("[%d] %d\n", jid, fgpid);
            fgpid = 0;
        }

        return;
    }
    free (token);
}

void tsh_prompt() {
    //const char *s_user = strdupa(getlogin());
    const char *s_user = getenv("USER");
    char s_host[1024];
    gethostname(s_host, 1024);

    char *prompt;
    char *homedir = realpath(getenv("HOME"), NULL);

    if (strncmp(homedir, getcwd(cwd, 1024), strlen(homedir)) == 0) {
        prompt = cwd + strlen(homedir) - 1;
        *prompt = '~';
    } else {
        prompt = cwd;
    }

    printf ("%s@%s:%s$ ", s_user, s_host, prompt);

    free(homedir);
}

int main(int argc, char *argv[]) {
    char line[1024];
    signal(SIGINT, sig_int);
    signal(SIGCHLD, sig_chld);
    signal(SIGTSTP, sig_tstp);
    signal(SIGTTOU, SIG_IGN); // don't stop!!!
    signal(SIGTTIN, SIG_IGN); // don't stop!!!
    setpgid(0, 0);
    main_pgrp = getpgrp();
    bg_list = p_init();

    for (;;) {
        tsh_prompt();
        if (fgets(line, 1024, stdin) == NULL ) {
            exit(0);
        }
        command(line);
    }

    return 0;
}
